<?php
/**
 * RAMBLER QATAR — Real PDF Flip Viewer
 * Renders actual PDF pages via PDF.js with StPageFlip page-turn animation.
 * URL: catalog-pdf-viewer.php?id=14
 */
$base     = '/ramblerco';
$cats_json = file_get_contents(__DIR__ . '/data/catalogs.json');
$data      = json_decode($cats_json, true);
$all_cats  = $data['catalogs'];

$cat_id  = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$catalog = null;
foreach ($all_cats as $c) {
  if ($c['id'] === $cat_id && ($c['viewer'] ?? '') === 'real-pdf') {
    $catalog = $c; break;
  }
}
// Redirect if not a real-pdf catalog
if (!$catalog) { header('Location: ' . $base . '/catalog.php'); exit; }

// Find prev/next real-pdf catalogs
$real_cats = array_values(array_filter($all_cats, fn($c) => ($c['viewer'] ?? '') === 'real-pdf'));
$cur_idx   = array_search($catalog, $real_cats);
$prev_cat  = $cur_idx > 0               ? $real_cats[$cur_idx - 1] : null;
$next_cat  = $cur_idx < count($real_cats) - 1 ? $real_cats[$cur_idx + 1] : null;

$pdf_url    = $base . '/' . $catalog['pdf_url'];
$gf         = $catalog['gradient_from'];
$gt         = $catalog['gradient_to'];
$ac         = $catalog['accent'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0">
  <title><?= htmlspecialchars($catalog['title']) ?> — Rambler Qatar</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link href="https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:wght@400;600;700;800&family=Inter:wght@400;500;600&display=swap" rel="stylesheet">

  <!-- PDF.js 3.11 (range-request aware — loads only needed bytes) -->
  <script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js"></script>
  <!-- StPageFlip -->
  <script src="https://unpkg.com/page-flip/dist/js/page-flip.browser.js"></script>

  <style>
    *, *::before, *::after { box-sizing: border-box; margin: 0; padding: 0; }
    html, body {
      height: 100%; overflow: hidden;
      background: #070409; color: #fff;
      font-family: 'Inter', sans-serif;
    }

    /* ══════════════════════════════
       TOP BAR
    ══════════════════════════════ */
    #topbar {
      position: fixed; top: 0; left: 0; right: 0; height: 56px; z-index: 100;
      display: flex; align-items: center; gap: 12px; padding: 0 16px 0 12px;
      background: rgba(7,4,9,.94); backdrop-filter: blur(24px);
      border-bottom: 1px solid rgba(255,255,255,.07);
    }
    #topbar .tb-logo { height: 30px; object-fit: contain; flex-shrink: 0; }
    #topbar .tb-sep  { width: 1px; height: 20px; background: rgba(255,255,255,.1); flex-shrink: 0; }
    #topbar .tb-title {
      font-family: 'Plus Jakarta Sans', sans-serif;
      font-size: 13px; font-weight: 700; color: rgba(255,255,255,.85);
      white-space: nowrap; overflow: hidden; text-overflow: ellipsis; flex: 1;
    }
    #topbar .tb-badge {
      padding: 3px 10px; border-radius: 100px; font-size: 10px; font-weight: 700;
      background: <?= $ac ?>22; border: 1px solid <?= $ac ?>44; color: <?= $ac ?>;
      flex-shrink: 0;
    }
    .ctrl-btn {
      display: inline-flex; align-items: center; justify-content: center;
      width: 36px; height: 36px; border-radius: 10px;
      border: 1px solid rgba(255,255,255,.1); background: transparent;
      color: rgba(255,255,255,.55); cursor: pointer; transition: all .2s;
      text-decoration: none; flex-shrink: 0;
    }
    .ctrl-btn:hover { border-color: rgba(255,255,255,.25); color: #fff; background: rgba(255,255,255,.06); }
    .open-pdf-btn {
      display: inline-flex; align-items: center; gap: 6px; padding: 0 14px; height: 36px;
      border-radius: 100px; font-size: 11px; font-weight: 700; letter-spacing: .04em;
      background: linear-gradient(135deg, <?= $ac ?>25, <?= $ac ?>10);
      border: 1px solid <?= $ac ?>40; color: <?= $ac ?>;
      text-decoration: none; white-space: nowrap; transition: all .25s; flex-shrink: 0;
    }
    .open-pdf-btn:hover { background: <?= $ac ?>35; box-shadow: 0 0 18px <?= $ac ?>30; }

    /* ══════════════════════════════
       STAGE
    ══════════════════════════════ */
    #stage {
      position: fixed; top: 56px; left: 0; right: 0; bottom: 60px;
      display: flex; align-items: center; justify-content: center;
      background:
        radial-gradient(ellipse at 50% 20%, <?= $ac ?>0A 0%, transparent 55%),
        radial-gradient(ellipse at 80% 80%, rgba(28,46,112,.06) 0%, transparent 50%),
        #070409;
    }
    #stage::before {
      content: ''; position: absolute; inset: 0; pointer-events: none;
      background-image:
        linear-gradient(rgba(255,255,255,.012) 1px, transparent 1px),
        linear-gradient(90deg, rgba(255,255,255,.012) 1px, transparent 1px);
      background-size: 50px 50px;
    }

    /* ══════════════════════════════
       LOADING SCREEN
    ══════════════════════════════ */
    #loader {
      position: absolute; inset: 0; z-index: 10;
      display: flex; flex-direction: column; align-items: center; justify-content: center;
      gap: 18px;
      background: #070409;
    }
    .loader-cover-preview {
      width: 120px; height: 165px; border-radius: 4px 10px 10px 4px; overflow: hidden;
      background: linear-gradient(135deg, <?= $gf ?>, <?= $gt ?>);
      box-shadow: 0 20px 60px rgba(0,0,0,.8);
      position: relative; flex-shrink: 0;
    }
    .loader-cover-preview canvas { position: absolute; inset: 0; width: 100%; height: 100%; object-fit: cover; }
    .loader-cover-preview .cover-loader-ring {
      position: absolute; inset: 0; display: flex; align-items: center; justify-content: center;
    }
    .loader-ring {
      width: 36px; height: 36px; border-radius: 50%;
      border: 3px solid rgba(255,255,255,.08); border-top-color: <?= $ac ?>;
      animation: spin .85s linear infinite;
    }
    @keyframes spin { to { transform: rotate(360deg); } }
    #loader-title {
      font-family: 'Plus Jakarta Sans', sans-serif;
      font-size: 15px; font-weight: 700; color: #fff; text-align: center;
    }
    #loader-sub { font-size: 11px; color: rgba(255,255,255,.35); letter-spacing: .07em; text-align: center; }
    .loader-progress-track {
      width: 220px; height: 2px; background: rgba(255,255,255,.07); border-radius: 2px; overflow: hidden;
    }
    #loader-progress-fill {
      height: 100%; width: 0%; background: linear-gradient(to right, <?= $ac ?>, #E03540);
      border-radius: 2px; transition: width .35s ease;
    }

    /* ══════════════════════════════
       BOOK WRAPPER + PAGES
    ══════════════════════════════ */
    #book-wrapper { display: none; position: relative; }

    /* StPageFlip page slots */
    .page-slot {
      overflow: hidden; user-select: none;
      background: linear-gradient(135deg, <?= $gf ?>, <?= $gt ?>);
    }
    /* Loading placeholder inside each page slot */
    .page-placeholder {
      width: 100%; height: 100%;
      display: flex; align-items: center; justify-content: center;
      background: linear-gradient(135deg, <?= $gf ?>ee, <?= $gt ?>ee);
    }
    .page-placeholder .mini-ring {
      width: 24px; height: 24px; border-radius: 50%;
      border: 2px solid rgba(255,255,255,.1); border-top-color: <?= $ac ?>aa;
      animation: spin .8s linear infinite;
    }
    /* Rendered canvas fills the page slot */
    .page-slot canvas {
      display: block;
      /* Force canvas to fill parent; actual px set by JS */
      width: 100% !important;
      height: 100% !important;
      object-fit: cover;
    }

    /* ══════════════════════════════
       BOTTOM CONTROLS
    ══════════════════════════════ */
    #controls {
      position: fixed; bottom: 0; left: 0; right: 0; height: 60px; z-index: 100;
      display: flex; align-items: center; justify-content: center; gap: 8px;
      padding: 0 16px;
      background: rgba(7,4,9,.92); backdrop-filter: blur(24px);
      border-top: 1px solid rgba(255,255,255,.07);
    }
    .nav-btn {
      width: 36px; height: 36px; display: flex; align-items: center; justify-content: center;
      border-radius: 10px; border: 1px solid rgba(255,255,255,.1); background: transparent;
      color: rgba(255,255,255,.5); cursor: pointer; transition: all .2s;
    }
    .nav-btn:hover  { background: rgba(255,255,255,.07); color: #fff; }
    .nav-btn:disabled { opacity: .25; cursor: default; }
    #page-indicator {
      font-size: 11px; font-weight: 500; color: rgba(255,255,255,.4);
      min-width: 96px; text-align: center; letter-spacing: .04em;
    }
    .dl-btn {
      display: inline-flex; align-items: center; gap: 5px;
      padding: 0 12px; height: 36px; border-radius: 10px;
      border: 1px solid rgba(255,255,255,.1); background: transparent;
      color: rgba(255,255,255,.5); font-size: 11px; font-weight: 600;
      text-decoration: none; transition: all .2s; white-space: nowrap;
    }
    .dl-btn:hover { background: rgba(255,255,255,.07); color: #fff; }
    .quote-btn {
      display: inline-flex; align-items: center; gap: 5px;
      padding: 0 14px; height: 36px; border-radius: 10px;
      background: linear-gradient(135deg, #FF4D5A, #C42830);
      color: #fff; font-size: 11px; font-weight: 700;
      text-decoration: none; white-space: nowrap; transition: box-shadow .2s;
    }
    .quote-btn:hover { box-shadow: 0 4px 20px rgba(196,40,48,.45); }
    .cat-nav-link {
      font-size: 10px; font-weight: 600; color: rgba(255,255,255,.28);
      text-decoration: none; white-space: nowrap; letter-spacing: .04em;
      transition: color .2s; padding: 0 4px;
    }
    .cat-nav-link:hover { color: rgba(255,255,255,.7); }

    /* ══════════════════════════════
       PROGRESS BAR
    ══════════════════════════════ */
    #read-progress {
      position: fixed; bottom: 60px; left: 0; height: 2px; z-index: 101;
      background: linear-gradient(to right, <?= $ac ?>, #E03540);
      width: 0%; transition: width .45s cubic-bezier(.4,0,.2,1);
    }

    /* ══════════════════════════════
       KEYBOARD HINT
    ══════════════════════════════ */
    #key-hint {
      position: fixed; bottom: 72px; right: 16px; z-index: 90;
      font-size: 9px; color: rgba(255,255,255,.2); letter-spacing: .08em;
      pointer-events: none;
      animation: fadeHint 3s 2s forwards;
    }
    @keyframes fadeHint { to { opacity: 0; } }
  </style>
</head>
<body>

<!-- ══ TOP BAR ════════════════════════════════════════════ -->
<div id="topbar">
  <a href="<?= $base ?>/catalog.php">
    <img src="<?= $base ?>/logo (2).png" class="tb-logo" alt="Rambler Qatar">
  </a>
  <div class="tb-sep"></div>
  <div class="tb-title"><?= htmlspecialchars($catalog['title']) ?></div>
  <div class="tb-badge"><?= $catalog['year'] ?></div>

  <div style="display:flex;align-items:center;gap:8px;margin-left:8px;">
    <!-- Open actual PDF in new tab -->
    <a href="<?= $pdf_url ?>" target="_blank" rel="noopener" class="open-pdf-btn">
      <svg style="width:12px;height:12px;flex-shrink:0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
              d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14"/>
      </svg>
      Open PDF
    </a>
    <!-- Download -->
    <a href="<?= $pdf_url ?>" download class="ctrl-btn" title="Download PDF">
      <svg style="width:15px;height:15px" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
              d="M12 10v6m0 0l-3-3m3 3l3-3M3 17v3a1 1 0 001 1h16a1 1 0 001-1v-3"/>
      </svg>
    </a>
    <!-- Close / back -->
    <a href="<?= $base ?>/catalog.php" class="ctrl-btn" title="All Catalogues">
      <svg style="width:15px;height:15px" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
      </svg>
    </a>
  </div>
</div>

<!-- ══ STAGE ═════════════════════════════════════════════ -->
<div id="stage">

  <!-- Loading screen -->
  <div id="loader">
    <div class="loader-cover-preview">
      <canvas id="cover-preview-canvas"></canvas>
      <div class="cover-loader-ring" id="cover-ring">
        <div class="loader-ring"></div>
      </div>
    </div>
    <div id="loader-title"><?= htmlspecialchars($catalog['title']) ?></div>
    <div id="loader-sub">Loading catalogue…</div>
    <div class="loader-progress-track">
      <div id="loader-progress-fill"></div>
    </div>
  </div>

  <!-- Flip book container (hidden until ready) -->
  <div id="book-wrapper">
    <div id="book"></div>
  </div>

</div>

<!-- ══ PROGRESS BAR ═══════════════════════════════════════ -->
<div id="read-progress"></div>

<!-- ══ BOTTOM CONTROLS ════════════════════════════════════ -->
<div id="controls">
  <?php if ($prev_cat): ?>
  <a href="?id=<?= $prev_cat['id'] ?>" class="cat-nav-link" style="display:none;" id="lnk-prev">
    ← <?= htmlspecialchars(mb_substr($prev_cat['title'], 0, 18)) ?>
  </a>
  <?php endif; ?>

  <!-- First page -->
  <button class="nav-btn" id="btn-first" title="First page" onclick="goFirst()">
    <svg style="width:13px;height:13px" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
            d="M11 19l-7-7 7-7m8 14l-7-7 7-7"/>
    </svg>
  </button>
  <!-- Prev -->
  <button class="nav-btn" id="btn-prev" title="Previous" onclick="goPrev()">
    <svg style="width:13px;height:13px" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
    </svg>
  </button>

  <span id="page-indicator">Loading…</span>

  <!-- Next -->
  <button class="nav-btn" id="btn-next" title="Next" onclick="goNext()">
    <svg style="width:13px;height:13px" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/>
    </svg>
  </button>
  <!-- Last page -->
  <button class="nav-btn" id="btn-last" title="Last page" onclick="goLast()">
    <svg style="width:13px;height:13px" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
            d="M13 5l7 7-7 7M5 5l7 7-7 7"/>
    </svg>
  </button>

  <a href="<?= $pdf_url ?>" download class="dl-btn" style="display:none;" id="lnk-dl">
    <svg style="width:12px;height:12px" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2"
            d="M12 10v6m0 0l-3-3m3 3l3-3M3 17v3a1 1 0 001 1h16a1 1 0 001-1v-3"/>
    </svg>
    Download PDF
  </a>
  <a href="<?= $base ?>/index.php#contact" class="quote-btn">Get Quote →</a>

  <?php if ($next_cat): ?>
  <a href="?id=<?= $next_cat['id'] ?>" class="cat-nav-link" style="display:none;" id="lnk-next">
    <?= htmlspecialchars(mb_substr($next_cat['title'], 0, 18)) ?> →
  </a>
  <?php endif; ?>
</div>

<div id="key-hint">← → ARROW KEYS TO TURN PAGES</div>

<!-- ══ SCRIPT ════════════════════════════════════════════ -->
<script>
// ── Config ─────────────────────────────────────────────
pdfjsLib.GlobalWorkerOptions.workerSrc =
  'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

const PDF_URL   = '<?= addslashes($pdf_url) ?>';
const GRAD_FROM = '<?= $gf ?>';
const GRAD_TO   = '<?= $gt ?>';
const ACCENT    = '<?= $ac ?>';

// ── State ───────────────────────────────────────────────
let pdfDoc        = null;
let pageFlip      = null;
let totalPages    = 0;
let pageW         = 0;
let pageH         = 0;
let renderScale   = 1.5;          // final render resolution scale
let canvasCache   = {};           // { pageNum: HTMLCanvasElement }
let renderQ       = [];           // pending page numbers
let qRunning      = false;

// ── Helpers ─────────────────────────────────────────────
function setProgress(pct) {
  document.getElementById('loader-progress-fill').style.width = pct + '%';
}
function setLoaderSub(msg) {
  document.getElementById('loader-sub').textContent = msg;
}
function showUI() {
  document.getElementById('loader').style.display = 'none';
  document.getElementById('book-wrapper').style.display = 'block';
  // Reveal previously hidden controls
  ['lnk-prev','lnk-dl','lnk-next'].forEach(id => {
    const el = document.getElementById(id);
    if (el) el.style.display = '';
  });
}

// ── Calculate page display size from viewport ────────────
function calcPageSize(pdfViewport) {
  const stage   = document.getElementById('stage');
  const sw      = stage.offsetWidth  - 80;  // horizontal padding
  const sh      = stage.offsetHeight - 40;  // vertical padding
  const pdfW    = pdfViewport.width;
  const pdfH    = pdfViewport.height;
  // Double-page spread: each page gets half the width
  const scaleW  = (sw / 2) / pdfW;
  const scaleH  = sh / pdfH;
  const scale   = Math.min(scaleW, scaleH, 2.2);
  return {
    w: Math.floor(pdfW * scale),
    h: Math.floor(pdfH * scale),
    renderScale: Math.min(scale * (window.devicePixelRatio || 1), 3.5)
  };
}

// ── Render one PDF page → JPEG data URL ──────────────────
// (Storing as data URL, not canvas, because canvas.cloneNode()
//  copies the DOM element but NOT the pixel buffer.)
async function renderPage(num) {
  if (canvasCache[num] || num < 1 || num > totalPages) return;
  const page     = await pdfDoc.getPage(num);
  const viewport = page.getViewport({ scale: renderScale });

  const canvas = document.createElement('canvas');
  canvas.width  = viewport.width;
  canvas.height = viewport.height;

  await page.render({ canvasContext: canvas.getContext('2d'), viewport }).promise;

  // Store as data URL (JPEG 88% quality — good balance size/sharpness)
  canvasCache[num] = canvas.toDataURL('image/jpeg', 0.88);

  // Also keep a reference for the loader cover preview (page 1 only)
  if (num === 1) canvasCache._firstCanvas = canvas;

  // If flip is live, inject into the matching slot
  if (pageFlip) injectCanvas(num);
}

// ── Inject rendered data URL as <img> into a page slot ───
function injectCanvas(num) {
  const slot = document.querySelector('#book .page-slot[data-pn="' + num + '"]');
  if (!slot || !canvasCache[num]) return;
  slot.innerHTML = '';
  const img = document.createElement('img');
  img.src           = canvasCache[num];
  img.style.width   = pageW + 'px';
  img.style.height  = pageH + 'px';
  img.style.display = 'block';
  img.draggable     = false;
  slot.appendChild(img);
  pageFlip.update();  // re-snapshot so StPageFlip picks up the new image
}

// ── Priority render queue ────────────────────────────────
function enqueue(pages) {
  pages.forEach(p => {
    if (p >= 1 && p <= totalPages && !canvasCache[p] && !renderQ.includes(p))
      renderQ.push(p);
  });
  drainQueue();
}
async function drainQueue() {
  if (qRunning || renderQ.length === 0) return;
  qRunning = true;
  while (renderQ.length > 0) {
    await renderPage(renderQ.shift());
  }
  qRunning = false;
}

// ── Build StPageFlip with placeholder slots ─────────────
function buildFlip() {
  const book = document.getElementById('book');
  book.innerHTML = '';

  for (let p = 1; p <= totalPages; p++) {
    const slot = document.createElement('div');
    slot.className = 'page-slot';
    slot.dataset.pn = p;
    slot.style.width  = pageW + 'px';
    slot.style.height = pageH + 'px';

    if (canvasCache[p]) {
      // Already rendered — insert as <img> from data URL
      const img = document.createElement('img');
      img.src           = canvasCache[p];
      img.style.width   = pageW + 'px';
      img.style.height  = pageH + 'px';
      img.style.display = 'block';
      img.draggable     = false;
      slot.appendChild(img);
    } else {
      slot.innerHTML =
        '<div class="page-placeholder"><div class="mini-ring"></div></div>';
    }
    book.appendChild(slot);
  }

  pageFlip = new St.PageFlip(book, {
    width           : pageW,
    height          : pageH,
    size            : 'fixed',
    showCover       : true,
    maxShadowOpacity: 0.6,
    drawShadow      : true,
    flippingTime    : 750,
    usePortrait     : false,
    mobileScrollSupport: false,
    clickEventForward: false,
    startZIndex     : 20,
    autoSize        : false,
  });

  pageFlip.loadFromHTML(book.querySelectorAll('.page-slot'));

  // Events
  pageFlip.on('flip', e => {
    const cur = e.data + 1;
    updateIndicator(cur);
    // Pre-render surrounding spread
    enqueue([cur-2, cur-1, cur, cur+1, cur+2, cur+3, cur+4]);
  });

  pageFlip.on('changeState', () => updateProgress());
}

// ── UI updates ───────────────────────────────────────────
function updateIndicator(pageNum) {
  document.getElementById('page-indicator').textContent =
    'Page ' + pageNum + ' of ' + totalPages;
  updateProgress(pageNum);
}
function updateProgress(pageNum) {
  if (!pageFlip && !pageNum) return;
  const cur = pageNum || (pageFlip ? pageFlip.getCurrentPageIndex() + 1 : 1);
  const pct = totalPages > 1 ? ((cur - 1) / (totalPages - 1)) * 100 : 0;
  document.getElementById('read-progress').style.width = pct + '%';
}

// ── Nav controls ─────────────────────────────────────────
function goNext()  { if (pageFlip) pageFlip.flipNext(); }
function goPrev()  { if (pageFlip) pageFlip.flipPrev(); }
function goFirst() { if (pageFlip) pageFlip.flip(0); }
function goLast()  { if (pageFlip) pageFlip.flip(totalPages - 1); }

// ── Main loader ──────────────────────────────────────────
async function init() {
  try {
    // PDF.js uses HTTP range requests — only fetches bytes it needs
    const loadTask = pdfjsLib.getDocument({
      url            : PDF_URL,
      rangeChunkSize : 65536,   // 64 KB per chunk
      disableAutoFetch: true,   // don't pre-load entire file
      disableStream  : false,
    });

    loadTask.onProgress = ({ loaded, total }) => {
      if (total > 0) setProgress(Math.min((loaded / total) * 30, 30));
    };

    pdfDoc     = await loadTask.promise;
    totalPages = pdfDoc.numPages;
    setProgress(35);
    setLoaderSub(totalPages + ' pages detected — rendering cover…');

    // Determine page display size from page 1 dimensions
    const pg1       = await pdfDoc.getPage(1);
    const vp1       = pg1.getViewport({ scale: 1 });
    const sizing    = calcPageSize(vp1);
    pageW           = sizing.w;
    pageH           = sizing.h;
    renderScale     = sizing.renderScale;

    setProgress(45);

    // Render page 1 → show in loader cover preview
    await renderPage(1);
    const prevCanvas = document.getElementById('cover-preview-canvas');
    if (canvasCache._firstCanvas && prevCanvas) {
      prevCanvas.width  = canvasCache._firstCanvas.width;
      prevCanvas.height = canvasCache._firstCanvas.height;
      prevCanvas.getContext('2d').drawImage(canvasCache._firstCanvas, 0, 0);
      document.getElementById('cover-ring').style.display = 'none';
    }

    setProgress(60);
    setLoaderSub('Rendering first spread…');

    // Render page 2 before showing flip (so first spread looks complete)
    await renderPage(2);
    setProgress(80);

    // Build StPageFlip
    buildFlip();
    setProgress(100);
    setLoaderSub('Ready!');

    // Small delay for smooth transition
    await new Promise(r => setTimeout(r, 320));
    showUI();
    updateIndicator(1);

    // Background: render pages 3-10 lazily
    enqueue([3, 4, 5, 6, 7, 8, 9, 10].filter(p => p <= totalPages));

  } catch (err) {
    console.error('PDF load error:', err);
    setLoaderSub('Error loading PDF — try refreshing.');
    document.querySelector('.loader-ring').style.borderTopColor = '#E03540';
    document.getElementById('loader-progress-fill').style.background = '#E03540';
  }
}

// ── Keyboard navigation ──────────────────────────────────
document.addEventListener('keydown', e => {
  if (!pageFlip) return;
  if (e.key === 'ArrowRight' || e.key === 'ArrowDown'  || e.key === ' ') goNext();
  if (e.key === 'ArrowLeft'  || e.key === 'ArrowUp')                      goPrev();
  if (e.key === 'Home') goFirst();
  if (e.key === 'End')  goLast();
});

// ── Start ────────────────────────────────────────────────
init();
</script>
</body>
</html>
